﻿document.addEventListener("DOMContentLoaded", function () {

  function observeShippingChanges() {
    const container = document.getElementById('checkout-shipping-method-load');
    if (!container) return;

    if (container.dataset.packetaInitObserverAttached === '1') return;
    container.dataset.packetaInitObserverAttached = '1';

    let _packetaInitTimer = null;

    const observer = new MutationObserver(() => {
      if (!container.querySelector('input[name="shippingoption"]')) return;

      clearTimeout(_packetaInitTimer);
      _packetaInitTimer = setTimeout(() => runPacketaInit(), 50);
    });

    observer.observe(container, { childList: true, subtree: true });

    runPacketaInit();
  }

  let _packetaInitRunning = false;

  async function runPacketaInit() {
    if (_packetaInitRunning) return;
    _packetaInitRunning = true;

    try {
      await loadPacketaMetaOnce();

      const container = document.getElementById('checkout-shipping-method-load');
      if (!container) return;

      initPacketaUI(container);

      const selected = container.querySelector('input[name="shippingoption"]:checked');
      if (!selected) {
        updateNextButtonState(null);
        return;
      }

      const { system } = parseShippingValue(selected.value);
      const isPacketa = system.toLowerCase().includes('shipping.packeta');
      const isPickup = isPacketaPickup(selected.value);

      if (isPacketa && !isPickup) {
        handlePacketaSelection(selected);
      } else {
        updateNextButtonState(selected);
      }
    }
    finally {
      _packetaInitRunning = false;
    }
  }

  let PACKETA_META = new Map();
  let _packetaMetaPromise = null;

  function parseShippingValue(value) {
    const [name, system] = String(value || '').split('___');
    return { name: (name || '').trim(), system: (system || '').trim() };
  }

  function isPacketaPickup(value) {
    const { name, system } = parseShippingValue(value);
    if (!system.toLowerCase().includes('shipping.packeta')) return false;
    const meta = PACKETA_META.get(name.toLowerCase());
    return !!(meta && meta.pickup);
  }

  async function loadPacketaMetaOnce() {
    if (_packetaMetaPromise) return _packetaMetaPromise;

    _packetaMetaPromise = (async () => {
      let json = {};
      try {
        const r = await fetch('/PacketaCheckout/GetPacketaOptionTypes', { cache: 'no-store' });
        json = r.ok ? await r.json() : {};
      } catch {
        json = {};
      }

      const map = new Map();
      Object.entries(json).forEach(([key, val]) => {
        const k = String(key).toLowerCase().trim();

        if (val && typeof val === 'object') {
          const pickup =
            (Object.prototype.hasOwnProperty.call(val, 'pickup') ? val.pickup :
              Object.prototype.hasOwnProperty.call(val, 'Pickup') ? val.Pickup :
                false);

          const reqDrop =
            (Object.prototype.hasOwnProperty.call(val, 'RequireDropoffCapability') ? val.RequireDropoffCapability :
              Object.prototype.hasOwnProperty.call(val, 'requireDropoffCapability') ? val.requireDropoffCapability :
                true);

          map.set(k, { pickup: !!pickup, RequireDropoffCapability: !!reqDrop });
        } else {
          map.set(k, { pickup: !!val, RequireDropoffCapability: true });
        }
      });

      PACKETA_META = map;
      return PACKETA_META;
    })();

    return _packetaMetaPromise;
  }

  function setNextDisabledFlag(flagName, shouldDisable) {
    const btn = document.querySelector('.shipping-method-next-step-button');
    if (!btn) return;
    if (shouldDisable) btn.setAttribute(flagName, '1');
    else btn.removeAttribute(flagName);
    recomputeNextDisabled();
  }

  function recomputeNextDisabled() {
    const btn = document.querySelector('.shipping-method-next-step-button');
    if (!btn) return;
    const anyDisabled = Array.from(btn.attributes).some(a => a.name.startsWith('data-next-disabled-') && a.value === '1');
    btn.disabled = anyDisabled;
  }

  function clearOtherPickupAddresses(selectedLi) {
    document.querySelectorAll('.packeta-selected-address, .ppl-selected-address, .dpd-selected-address')
      .forEach(el => { if (!selectedLi || !selectedLi.contains(el)) el.remove(); });
  }

  function updateNextButtonState(selectedRadio) {
    const selected = selectedRadio || document.querySelector('input[name="shippingoption"]:checked');

    if (!selected) {
      setNextDisabledFlag('data-next-disabled-packeta', false);
      return;
    }

    const { system } = parseShippingValue(selected.value);

    if (!system.toLowerCase().includes('shipping.packeta')) {
      setNextDisabledFlag('data-next-disabled-packeta', false);
      return;
    }

    const li = selected.closest('li');
    if (!li) {
      setNextDisabledFlag('data-next-disabled-packeta', false);
      return;
    }

    const isPickup = isPacketaPickup(selected.value);
    if (isPickup) {
      const hasAddress = !!li.querySelector('.packeta-selected-address');
      setNextDisabledFlag('data-next-disabled-packeta', !hasAddress);
    } else {
      setNextDisabledFlag('data-next-disabled-packeta', false);
    }
  }

  function handlePacketaSelection(radio) {
    if (!radio) return;

    const selectedLi = radio.closest('li');
    clearOtherPickupAddresses(selectedLi);

    const isPickup = isPacketaPickup(radio.value);
    const { system } = parseShippingValue(radio.value);
    const isPacketa = system.toLowerCase().includes('shipping.packeta');

    if (isPacketa && !isPickup) {
      setNextDisabledFlag('data-next-disabled-packeta', false);

      const alreadySaved = radio.dataset.packetaAddressSaved === '1';
      const savingNow = radio.dataset.packetaAddressSaving === '1';

      if (!alreadySaved && !savingNow) {
        radio.dataset.packetaAddressSaving = '1';

        fetch('/PacketaCheckout/SaveShippingAddressAsPickupPoint')
          .then(r => {
            if (!r.ok) {
              console.warn('Packeta: SaveShippingAddressAsPickupPoint failed', r.status);
              delete radio.dataset.packetaAddressSaving;
              return;
            }
            radio.dataset.packetaAddressSaved = '1';
            delete radio.dataset.packetaAddressSaving;
          })
          .catch(err => {
            console.error('Packeta SaveShippingAddressAsPickupPoint error:', err);
            delete radio.dataset.packetaAddressSaving;
          });
      }
    }

    updateNextButtonState(radio);
  }

  function initPacketaUI(scope) {
    const WIDGET_MODE = Number(window.packetaCheckoutWidgetMode || 0);
    const ICON_URL = window.packetaIconUrl || "";

    if (!document.getElementById('packeta-disabled-style')) {
      const style = document.createElement('style');
      style.id = 'packeta-disabled-style';
      style.textContent = `
        .packeta-pickup-button:disabled {
          background: #ccc !important;
          color: #666 !important;
          cursor: not-allowed !important;
        }
        .method-name.packeta-with-icon { position: relative; padding-left: 56px; }
        .method-name .packeta-method-icon-left {
          position: absolute; left: 12px; top: 50%; transform: translateY(-50%);
          width: 42px; height: 42px; object-fit: contain; pointer-events: none;
        }
        .shipping-method-next-step-button[disabled] { opacity: .6; cursor: not-allowed; pointer-events: none; }
      `;
      document.head.appendChild(style);
    }

    const radios = (scope || document).querySelectorAll('input[type="radio"][name="shippingoption"]');
    radios.forEach(radio => {
      const li = radio.closest('li');
      if (!li || li.dataset.packetaEnhanced === '1') return;

      const { system } = parseShippingValue(radio.value);
      const isPacketa = system.toLowerCase().includes('shipping.packeta');

      if (isPacketa && WIDGET_MODE === 1 && ICON_URL) {
        const methodName = li.querySelector('.method-name');
        if (methodName) {
          methodName.classList.add('packeta-with-icon');
          methodName.querySelector('.packeta-method-icon-left')?.remove();
          const img = new Image();
          img.src = ICON_URL;
          img.alt = "Packeta";
          img.className = "packeta-method-icon-left";
          methodName.insertAdjacentElement('afterbegin', img);
        }
      }

      if (isPacketa && isPacketaPickup(radio.value) && !li.querySelector('.packeta-pickup-button')) {
        const button = document.createElement('button');
        button.innerText = 'Vybrat výdejní místo';
        button.className = 'packeta-pickup-button';
        button.disabled = !radio.checked;
        button.style.cssText = `
          margin-top: 10px; padding: 6px 12px; background: #cc0000; color: white;
          border: none; font-size: 0.9em; border-radius: 4px; cursor: pointer;
        `;
        button.addEventListener('click', e => { e.preventDefault(); openPacketaWidget(); });
        li.querySelector('.method-name')?.appendChild(button);
      }

      if (radio.dataset.packetaWired !== '1') {
        radio.dataset.packetaWired = '1';
        radio.addEventListener('change', () => {
          document.querySelectorAll('.packeta-pickup-button').forEach(btn => {
            const li2 = btn.closest('li');
            const input = li2?.querySelector('input[type="radio"][name="shippingoption"]');
            btn.disabled = !input?.checked;
          });

          handlePacketaSelection(radio);
        });
      }

      li.dataset.packetaEnhanced = '1';
    });

    const selected = document.querySelector('input[name="shippingoption"]:checked');
    updateNextButtonState(selected);
  }

  function getCurrentPacketConsignment() {
    let result = false;

    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (!selected) return result;

    const { name, system } = parseShippingValue(selected.value);
    if (!system.toLowerCase().includes('shipping.packeta')) return result;

    const meta = PACKETA_META.get(name.toLowerCase());
    if (!meta) return result;

    result = !Boolean(meta.RequireDropoffCapability);
    return result;
  }

  window.openPacketaWidget = function () {
    if (typeof Packeta?.Widget?.pick !== "function") {
      alert("Packeta widget není dostupný.");
      return;
    }

    const apiKey = window.packetaApiKey || '';
    const packetConsignment = getCurrentPacketConsignment();

    Packeta.Widget.pick(apiKey, function (point) {
      if (!point) return;

      const formatted = `${point.name}, ${point.street}, ${point.zip} ${point.city}, ${point.country}`;
      const data = new URLSearchParams();
      data.append('Id', point.id || point.pickupPointId || '');
      data.append('Name', point.place || point.name || '');
      data.append('City', point.city || '');
      data.append('Street', point.street || '');
      data.append('Zip', point.zip || '');
      data.append('Country', point.country || '');
      data.append('CarrierId', point.carrierId || '');
      data.append('CarrierPickupPointId', point.carrierPickupPointId || '');
      data.append('FormattedAddress', formatted);
      data.append('IsPickupPoint', 'true');

      fetch('/PacketaCheckout/SavePickupPoint', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: data.toString()
      })
        .then(r => {
          if (!r.ok) console.warn('Packeta: SavePickupPoint failed', r.status);
        })
        .catch(err => console.error('Packeta SavePickupPoint error:', err));

      const selectedRadio = document.querySelector('input[name="shippingoption"]:checked');
      const selectedLi = selectedRadio?.closest('li');

      if (selectedLi) {
        selectedLi.querySelector('.packeta-selected-address')?.remove();

        const addrDiv = document.createElement('div');
        addrDiv.className = 'packeta-selected-address';
        addrDiv.style.marginTop = '5px';
        addrDiv.style.fontSize = '0.9em';
        addrDiv.style.color = '#333';
        addrDiv.textContent = formatted;

        const btn = selectedLi.querySelector('.packeta-pickup-button');
        btn?.insertAdjacentElement('afterend', addrDiv);
      }

      if (selectedRadio) {
        selectedRadio.dataset.packetaAddressSaved = '1';
        delete selectedRadio.dataset.packetaAddressSaving;
      }

      updateNextButtonState(selectedRadio);
    }, {
      country: "cz,sk",
      language: "cs",
      valueFormat: "\"Packeta\",id,carrierId,carrierPickupPointId,name,city,street",
      view: "modal",
      packetConsignment: packetConsignment
    });
  };

  observeShippingChanges();
});
