﻿(function () {
  // ---- CONFIG ------------------------------------------------------------
  const ARES_ENDPOINT = '/AresForm/GetByIco';

  // ---- HELPERS -----------------------------------------------------------
  const q = (s, r) => (r || document).querySelector(s);
  const qa = (s, r) => Array.from((r || document).querySelectorAll(s));
  const deaccent = (s) => (s || '').toString()
    .normalize('NFD').replace(/[\u0300-\u036f]/g, '')
    .toLowerCase().trim();

  function getRegisterForm() {
    // 1) podle action
    return q("form[action*='register' i]") ||
      q("form[action*='customer/info' i]") ||
      // 2) podle tlačítek
      (q('#register-button') && q('#register-button').closest('form')) ||
      (q('#save-info-button') && q('#save-info-button').closest('form')) ||
      // 3) fallback
      q('form');
  }

  function findCompanyRow(form) {
    const el = q("input[name='Company'], #Company", form);
    return el ? el.closest('.inputs') : null;
  }

  function findVatRow(form) {
    const el = q("input[name='VatNumber'], #VatNumber", form);
    return el ? el.closest('.inputs') : null;
  }

  function findCompanyContainer(form) {
    const row = findCompanyRow(form) || findVatRow(form);
    return row ? row.closest('.form-fields') : null;
  }

  function findIcoRow(form) {
    let rows = qa('.inputs.custom-attributes', form);
    for (const row of rows) {
      const lbl = row.querySelector('label');
      if (lbl && deaccent(lbl.textContent).includes('ico')) return row;
    }
    rows = qa('.inputs', form);
    for (const row of rows) {
      const lbl = row.querySelector('label');
      if (lbl && deaccent(lbl.textContent).includes('ico')) return row;
    }
    return null;
  }

  function addAresButton(icoRow) {
    if (!icoRow || icoRow.querySelector('.ares-inline .ares-lookup-btn')) return;

    const icoInput = icoRow.querySelector('input, textarea, select');
    if (!icoInput) return;

    const sib = icoInput.nextElementSibling;
    const isValidation = sib && (sib.matches('.field-validation-valid, .field-validation-error') || sib.hasAttribute('data-valmsg-for'));

    let inline = icoInput.parentElement;
    if (!inline || !inline.classList.contains('ares-inline')) {
      inline = document.createElement('span');
      inline.className = 'ares-inline';

      if (isValidation) icoRow.insertBefore(inline, sib);
      else icoInput.insertAdjacentElement('afterend', inline);

      inline.appendChild(icoInput);

      while (inline.nextSibling && inline.nextSibling !== sib &&
        inline.nextSibling.nodeType === 1 &&
        inline.nextSibling.classList.contains('required')) {
        inline.appendChild(inline.nextSibling);
      }
    }

    const wrap = document.createElement('span');
    wrap.className = 'ares-lookup-wrap';

    const btn = document.createElement('button');
    btn.type = 'button';
    btn.className = 'ares-lookup-btn';
    btn.textContent = 'Vyhledat v ARES';

    wrap.appendChild(btn);
    inline.appendChild(wrap);

    const run = typeof runLookupFor === 'function'
      ? () => runLookupFor(icoRow.closest('.edit-address') || document, icoInput, btn)
      : () => runLookup(icoInput, btn);

    icoInput.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') { e.preventDefault(); run(); }
    });
    btn.addEventListener('click', run);
  }

  function ensureIcoUnderCompany() {
    const form = getRegisterForm(); if (!form) return false;

    const icoRow = findIcoRow(form); if (!icoRow) return false;
    addAresButton(icoRow);

    const container = findCompanyContainer(form); if (!container) return false;
    const companyRow = findCompanyRow(form);
    const vatRow = findVatRow(form);

    let correct = false;
    if (icoRow.closest('.form-fields') === container) {
      if (companyRow) {
        correct = (companyRow.nextElementSibling === icoRow);
      } else if (vatRow) {
        correct = (vatRow.previousElementSibling === icoRow);
      }
    }
    if (correct) { icoRow.dataset.aresMoved = '1'; return true; }

    if (companyRow) container.insertBefore(icoRow, companyRow.nextElementSibling);
    else if (vatRow) container.insertBefore(icoRow, vatRow);
    else container.appendChild(icoRow);

    icoRow.dataset.aresMoved = '1';
    return true;
  }

  function setIfExists(selector, value) {
    const form = getRegisterForm();
    const el = q(selector, form);
    if (!el) return;

    if (el.tagName === 'SELECT') {
      const v = (value || '').toString().trim();
      const opt = Array.from(el.options).find(o =>
        o.value.toString().trim().toLowerCase() === v.toLowerCase() ||
        o.textContent.trim().toLowerCase() === v.toLowerCase());
      if (opt) el.value = opt.value;
      el.dispatchEvent(new Event('change', { bubbles: true }));
      return;
    }

    el.value = value || '';
    el.dispatchEvent(new Event('input', { bubbles: true }));
    el.dispatchEvent(new Event('change', { bubbles: true }));
  }

  function fillFromAres(data) {
    const streetCombined = [data.street || '', data.houseNumber || ''].filter(Boolean).join(' ').trim();

    setIfExists("input[name='Company'], #Company", data.companyName);
    setIfExists("input[name='VatNumber'], #VatNumber", data.dic || '');

    setIfExists("input[name='StreetAddress'], #StreetAddress", streetCombined);
    setIfExists("input[name='City'], #City", data.city || '');
    setIfExists("input[name='ZipPostalCode'], #ZipPostalCode", data.zipPostalCode || '');

    if (data.countryId != null) {
      setIfExists("select[name='CountryId'], #CountryId", String(data.countryId));
    } else if (data.countryIso2) {
      trySetCountryByIso2("select[name='CountryId'], #CountryId", data.countryIso2);
    }
  }


  function toBool(v) {
    if (typeof v === 'boolean') return v;
    if (typeof v === 'string') return v.toLowerCase() === 'true';
    return !!v;
  }

  async function runLookup(icoInput, btn) {
    const form = getRegisterForm();
    const vatInput = findVatRow(form)?.querySelector('input');
    const ico = (icoInput.value || '').replace(/\s/g, '');
    const vat = vatInput ? (vatInput.value || '').trim() : '';

    const params = new URLSearchParams();
    if (ico) params.set('ico', ico);
    if (vat) params.set('vat', vat);

    const url = `${ARES_ENDPOINT}?${params.toString()}`;

    btn.disabled = true;
    btn.classList.add('is-loading');

    try {
      const res = await fetch(url, { method: 'GET', headers: { 'Accept': 'application/json' } });
      if (!res.ok) throw new Error('Chybná odpověď serveru');

      const json = await res.json();
      const success = toBool(json?.success ?? json?.Success);
      const data = json?.data ?? json?.Data;
      const message = json?.message ?? json?.Message ?? 'Subjekt nenalezen.';

      const looksLikeDto =
        data == null &&
        (json?.companyName || json?.CompanyName || json?.ico || json?.ICO);

      if (!success && !looksLikeDto) {
        alert(message);
        return;
      }

      const payload = data ?? json;
      fillFromAres(payload || {});
    } catch (e) {
      console.error(e);
      alert('Nepodařilo se načíst data z ARES. Zkuste to prosím znovu.');
    } finally {
      btn.disabled = false;
      btn.classList.remove('is-loading');
    }
  }

  function init() {
    ensureIcoUnderCompany();

    let tries = 0;
    const poll = setInterval(() => {
      tries++;
      if (ensureIcoUnderCompany() && tries > 4) { clearInterval(poll); }
      if (tries > 60) clearInterval(poll);
    }, 300);

    const mo = new MutationObserver(() => { ensureIcoUnderCompany(); });
    mo.observe(document.body, { childList: true, subtree: true });
  }

  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', init);
  else init();
})();