﻿document.addEventListener("DOMContentLoaded", function () {

  let PPL_META = new Map();
  let _pplMetaPromise = null;

  function parseShippingValue(value) {
    const [name, system] = String(value || '').split('___');
    return { name: (name || '').trim(), system: (system || '').trim() };
  }

  function isPplPickup(value) {
    const { name, system } = parseShippingValue(value);
    if (!system.toLowerCase().includes('shipping.ppl')) return false;
    const meta = PPL_META.get(name.toLowerCase());
    return !!(meta && meta.pickup);
  }

  async function loadPplMetaOnce() {
    if (_pplMetaPromise) return _pplMetaPromise;
    _pplMetaPromise = (async () => {
      let json = {};
      try {
        const r = await fetch('/PPLCheckout/GetPplOptionTypes', { cache: 'no-store' });
        json = r.ok ? await r.json() : {};
      } catch { json = {}; }

      const map = new Map();
      // zpětná kompatibilita: starý tvar { "name": true/false } -> pickup = value, allowBoxDelivery = true
      Object.entries(json).forEach(([key, val]) => {
        const k = String(key).toLowerCase().trim();
        if (val && typeof val === 'object') {
          map.set(k, { pickup: !!val.pickup, allowBoxDelivery: !!val.allowBoxDelivery });
        } else {
          map.set(k, { pickup: !!val, allowBoxDelivery: true });
        }
      });
      PPL_META = map;
      return PPL_META;
    })();
    return _pplMetaPromise;
  }

  function setNextDisabledFlag(flagName, shouldDisable) {
    const btn = document.querySelector('.shipping-method-next-step-button');
    if (!btn) return;
    if (shouldDisable) btn.setAttribute(flagName, '1');
    else btn.removeAttribute(flagName);
    recomputeNextDisabled();
  }

  function recomputeNextDisabled() {
    const btn = document.querySelector('.shipping-method-next-step-button');
    if (!btn) return;
    const anyDisabled = Array.from(btn.attributes)
      .some(a => a.name.startsWith('data-next-disabled-') && a.value === '1');
    btn.disabled = anyDisabled;
  }

  function clearOtherPickupAddresses(selectedLi) {
    document.querySelectorAll('.packeta-selected-address, .ppl-selected-address, .dpd-selected-address')
      .forEach(el => { if (!selectedLi || !selectedLi.contains(el)) el.remove(); });
  }

  function updateNextButtonState() {
    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (!selected) { setNextDisabledFlag('data-next-disabled-ppl', false); return; }

    const li = selected.closest('li');
    if (!li) { setNextDisabledFlag('data-next-disabled-ppl', false); return; }

    const isPickup = isPplPickup(selected.value);
    if (isPickup) {
      const hasAddress = !!li.querySelector('.ppl-selected-address');
      setNextDisabledFlag('data-next-disabled-ppl', !hasAddress);
    } else {
      setNextDisabledFlag('data-next-disabled-ppl', false);
    }
  }

  function shouldHideBoxesForCurrentOption() {
    let allow = true;
    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (selected) {
      const { name, system } = parseShippingValue(selected.value);
      if (system.toLowerCase().includes('shipping.ppl')) {
        const meta = PPL_META.get(name.toLowerCase());
        if (meta) allow = !!meta.allowBoxDelivery;
      }
    }
    return !allow;
  }

  function handlePplSelection(radio) {
    if (!radio) return;

    const selectedLi = radio.closest("li");
    clearOtherPickupAddresses(selectedLi);

    const isPickup = isPplPickup(radio.value);
    const { system } = parseShippingValue(radio.value);
    const isPpl2 = system.toLowerCase().includes('shipping.ppl');

    if (isPpl2 && !isPickup) {
      if (radio.dataset.pplAddressSaved === '1')
        return;

      fetch("/PPLCheckout/SaveShippingAddressAsPickupPoint")
        .then(r => {
          if (r.ok) {
            radio.dataset.pplAddressSaved = '1';
          }
        })
        .catch(err => console.error("PPL SaveShippingAddressAsPickupPoint error:", err));
    }

    updateNextButtonState();
  }

  function initPplUI(scope) {
    const WIDGET_MODE = Number(window.pplCheckoutWidgetMode || 0); // 0=text, 1=text+ikonka
    const ICON_URL = window.pplIconUrl || "";

    if (!document.getElementById("ppl-disabled-style")) {
      const style = document.createElement("style");
      style.id = "ppl-disabled-style";
      style.textContent = `
        .ppl-pickup-button:disabled {
          background: #ccc !important; color: #666 !important; cursor: not-allowed !important;
        }
        .method-name.ppl-with-icon { position: relative; padding-left: 56px; }
        .method-name .ppl-method-icon-left {
          position: absolute; left: 12px; top: 50%; transform: translateY(-50%);
          width: 42px; height: 42px; object-fit: contain; pointer-events: none;
        }
        .shipping-method-next-step-button[disabled] { opacity: .6; cursor: not-allowed; pointer-events: none; }
      `;
      document.head.appendChild(style);
    }

    if (!document.getElementById("ppl-modal-overlay")) {
      const html = `
<div id="ppl-modal-overlay" style="display:none;position:fixed;z-index:9999;top:0;left:0;width:100vw;height:100vh;background-color:rgba(0,0,0,0.5);justify-content:center;align-items:center;">
  <div style="position:relative;width:95%;height:95%;background:white;border-radius:6px;overflow:hidden;">
    <button id="ppl-modal-close" style="position:absolute;top:10px;right:10px;z-index:2;">Zavřít</button>
    <div id="ppl-parcelshop-map-host" style="width:100%;height:100%;"></div>
  </div>
</div>`;
      document.body.insertAdjacentHTML("beforeend", html);

      document.getElementById("ppl-modal-close").addEventListener("click", () => {
        document.getElementById("ppl-modal-overlay").style.display = "none";
      });

      if (!document.getElementById("ppl-map-css")) {
        const link = document.createElement("link");
        link.id = "ppl-map-css";
        link.rel = "stylesheet";
        link.href = "https://www.ppl.cz/sources/map/main.css";
        document.head.appendChild(link);
      }
    }

    const radios = (scope || document).querySelectorAll('input[type="radio"][name="shippingoption"]');
    radios.forEach((radio) => {
      const li = radio.closest("li");
      if (!li || li.dataset.pplEnhanced === '1') return;

      const { system } = parseShippingValue(radio.value);
      const isPpl = system.toLowerCase().includes('shipping.ppl');

      if (isPpl && WIDGET_MODE === 1 && ICON_URL) {
        const methodName = li.querySelector('.method-name');
        if (methodName) {
          methodName.classList.add('ppl-with-icon');
          methodName.querySelector('.ppl-method-icon-left')?.remove();
          const img = new Image();
          img.src = ICON_URL;
          img.alt = "PPL";
          img.className = "ppl-method-icon-left";
          methodName.insertAdjacentElement('afterbegin', img);
        }
      }

      if (isPpl && isPplPickup(radio.value) && !li.querySelector(".ppl-pickup-button")) {
        const button = document.createElement("button");
        button.innerText = "Vybrat výdejní místo";
        button.className = "ppl-pickup-button";
        button.disabled = !radio.checked;
        button.style.cssText = `
          margin-top: 10px; padding: 6px 12px; background: #00519e; color: white;
          border: none; font-size: 0.9em; border-radius: 4px; cursor: pointer;
        `;
        button.addEventListener("click", (e) => {
          e.preventDefault();

          const host = document.getElementById("ppl-parcelshop-map-host");
          if (host) {
            host.innerHTML = "";
            const mapDiv = document.createElement("div");
            mapDiv.id = "ppl-parcelshop-map";
            mapDiv.style.width = "100%";
            mapDiv.style.height = "100%";

            if (shouldHideBoxesForCurrentOption()) {
              mapDiv.setAttribute("data-hiddenpoints", "ParcelBox, AlzaBox");
            }

            host.appendChild(mapDiv);
          }

          const OLD_ID = "ppl-map-script";
          document.getElementById(OLD_ID)?.remove();

          const s = document.createElement("script");
          s.id = OLD_ID;
          s.src = "https://www.ppl.cz/sources/map/main.js?v=" + Date.now();
          s.onload = () => {
            document.getElementById("ppl-modal-overlay").style.display = "flex";
          };
          s.onerror = () => {
            console.error("PPL widget script failed to load");
            alert("PPL widget se nepodařilo načíst.");
          };
          document.head.appendChild(s);
        });
        li.querySelector(".method-name")?.appendChild(button);
      }

      if (radio.dataset.pplWired !== '1') {
        radio.dataset.pplWired = '1';
        radio.addEventListener("change", () => {
          document.querySelectorAll(".ppl-pickup-button").forEach((btn) => {
            const rli = btn.closest("li");
            const rinput = rli?.querySelector('input[type="radio"][name="shippingoption"]');
            btn.disabled = !rinput?.checked;
          });

          handlePplSelection(radio);
        });
      }

      li.dataset.pplEnhanced = '1';
    });

    updateNextButtonState();
  }

  function whenShippingOptionsReadyOnce(timeoutMs = 8000) {
    if (document.querySelector('input[name="shippingoption"]')) return Promise.resolve();
    return new Promise(resolve => {
      const target = document.getElementById('checkout-shipping-method-load') || document.body;
      const to = setTimeout(() => { obs.disconnect(); resolve(); }, timeoutMs);
      const obs = new MutationObserver(() => {
        if (document.querySelector('input[name="shippingoption"]')) {
          clearTimeout(to);
          obs.disconnect();
          resolve();
        }
      });
      obs.observe(target, { childList: true, subtree: true });
    });
  }

  const opcContainer = document.getElementById('checkout-shipping-method-load');
  if (opcContainer && !opcContainer.dataset.pplObserveAttached) {
    opcContainer.dataset.pplObserveAttached = '1';
    const mo = new MutationObserver(() => { initPplUI(opcContainer); });
    mo.observe(opcContainer, { childList: true, subtree: true });
  }

  (async () => {
    await loadPplMetaOnce();
    await whenShippingOptionsReadyOnce();
    initPplUI(document);

    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (selected) {
      const { system } = parseShippingValue(selected.value);
      const isPpl = system.toLowerCase().includes('shipping.ppl');
      const isPickup = isPplPickup(selected.value);

      if (isPpl && !isPickup) {
        handlePplSelection(selected);
      } else {
        updateNextButtonState();
      }
    } else {
      updateNextButtonState();
    }
  })();

  document.addEventListener("ppl-parcelshop-map", (event) => {
    const pt = event?.detail;
    if (!pt || !pt.code) return;

    document.getElementById("ppl-modal-overlay").style.display = "none";

    const formattedAddress = `${pt.accessPointType ?? ''} ${pt.city}, ${pt.street}, ${pt.zipCode}, ${pt.country ?? ''}`.trim();

    document.querySelectorAll(".ppl-pickup-button").forEach((btn) => {
      const li = btn.closest("li");
      const radio = li?.querySelector('input[type="radio"]');
      if (radio?.checked) {
        li.querySelector(".ppl-selected-address")?.remove();
        const addr = document.createElement("div");
        addr.className = "ppl-selected-address";
        addr.style.marginTop = "5px";
        addr.style.fontSize = "0.9em";
        addr.style.color = "#333";
        addr.textContent = formattedAddress;
        btn.insertAdjacentElement("afterend", addr);
      }
    });

    updateNextButtonState();

    const pickupPointModel = {
      Id: pt.code,
      Name: pt.name,
      Street: pt.street,
      City: pt.city,
      Zip: pt.zipCode,
      Country: pt.country,
      FormattedAddress: formattedAddress,
      CarrierId: 0,
      CarrierPickupPointId: pt.code,
      IsPickupPoint: true
    };

    const formData = new URLSearchParams();
    for (const key in pickupPointModel) formData.append(key, pickupPointModel[key]);

    fetch("/PPLCheckout/SavePickupPoint", {
      method: "POST",
      headers: { "Content-Type": "application/x-www-form-urlencoded" },
      body: formData.toString(),
    });

    let hidden = document.querySelector('input[name="PplSelectedPickupPoint"]');
    if (!hidden) {
      const form = document.querySelector('form[action*="checkout"]');
      if (form) {
        hidden = document.createElement('input');
        hidden.type = 'hidden';
        hidden.name = 'PplSelectedPickupPoint';
        form.appendChild(hidden);
      }
    }
    if (hidden) hidden.value = JSON.stringify(pickupPointModel);
  });

});