﻿document.addEventListener("DOMContentLoaded", function () {

  let DPD_META = new Map();
  let _dpdMetaPromise = null;

  function parseShippingValue(value) {
    const [name, system] = String(value || '').split('___');
    return { name: (name || '').trim(), system: (system || '').trim() };
  }

  function isDpdPickup(value) {
    const { name, system } = parseShippingValue(value);
    if (!system.toLowerCase().includes('shipping.dpd')) return false;
    const meta = DPD_META.get(name.toLowerCase());
    return !!(meta && meta.pickup);
  }

  async function loadDpdMetaOnce() {
    if (_dpdMetaPromise) return _dpdMetaPromise;

    _dpdMetaPromise = (async () => {
      let json = {};
      try {
        const r = await fetch('/DPDCheckout/GetDpdOptionTypes', { cache: 'no-store' });
        json = r.ok ? await r.json() : {};
      } catch {
        json = {};
      }

      const map = new Map();

      Object.entries(json).forEach(([key, val]) => {
        const k = String(key).toLowerCase().trim();

        if (val && typeof val === 'object') {
          const pickup =
            (Object.prototype.hasOwnProperty.call(val, 'pickup') ? val.pickup :
              Object.prototype.hasOwnProperty.call(val, 'Pickup') ? val.Pickup :
                false);

          const allowBox =
            (Object.prototype.hasOwnProperty.call(val, 'allowBoxDelivery') ? val.allowBoxDelivery :
              Object.prototype.hasOwnProperty.call(val, 'AllowBoxDelivery') ? val.AllowBoxDelivery :
                true);

          map.set(k, {
            pickup: !!pickup,
            allowBoxDelivery: !!allowBox
          });
        } else {
          map.set(k, { pickup: !!val, allowBoxDelivery: true });
        }
      });

      DPD_META = map;
      return DPD_META;
    })();

    return _dpdMetaPromise;
  }

  function setNextDisabledFlag(flagName, shouldDisable) {
    const btn = document.querySelector('.shipping-method-next-step-button');
    if (!btn) return;
    if (shouldDisable) btn.setAttribute(flagName, '1');
    else btn.removeAttribute(flagName);
    recomputeNextDisabled();
  }

  function recomputeNextDisabled() {
    const btn = document.querySelector('.shipping-method-next-step-button');
    if (!btn) return;
    const anyDisabled = Array.from(btn.attributes)
      .some(a => a.name.startsWith('data-next-disabled-') && a.value === '1');
    btn.disabled = anyDisabled;
  }

  function clearOtherPickupAddresses(selectedLi) {
    document.querySelectorAll('.packeta-selected-address, .ppl-selected-address, .dpd-selected-address')
      .forEach(el => { if (!selectedLi || !selectedLi.contains(el)) el.remove(); });
  }

  function updateNextButtonState() {
    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (!selected) { setNextDisabledFlag('data-next-disabled-dpd', false); return; }

    const li = selected.closest('li');
    if (!li) { setNextDisabledFlag('data-next-disabled-dpd', false); return; }

    const isPickup = isDpdPickup(selected.value);
    if (isPickup) {
      const hasAddress = !!li.querySelector('.dpd-selected-address');
      setNextDisabledFlag('data-next-disabled-dpd', !hasAddress);
    } else {
      setNextDisabledFlag('data-next-disabled-dpd', false);
    }
  }

  function handleDpdSelection(radio) {
    if (!radio) return;

    const selectedLi = radio.closest('li');
    clearOtherPickupAddresses(selectedLi);

    const isPickup = isDpdPickup(radio.value);
    const { system } = parseShippingValue(radio.value);
    const isDpd2 = system.toLowerCase().includes('shipping.dpd');

    if (isDpd2 && !isPickup) {
      // ochrana, ať to neposíláme víckrát pro stejný input
      if (radio.dataset.dpdAddressSaved === '1')
        return;

      fetch('/DPDCheckout/SaveShippingAddressAsPickupPoint')
        .then(r => r.json())
        .then(res => {
          if (!res?.success) {
            console.warn('Chyba při ukládání DPD adresy:', res);
          } else {
            radio.dataset.dpdAddressSaved = '1';
          }
        })
        .catch(err => console.error('DPD address save error:', err));
    }

    updateNextButtonState();
  }

  function initDpdUI(scope) {
    const WIDGET_MODE = Number(window.dpdCheckoutWidgetMode || 0);
    const ICON_URL = window.dpdIconUrl || "";

    if (!document.getElementById('dpd-disabled-style')) {
      const style = document.createElement('style');
      style.id = 'dpd-disabled-style';
      style.textContent = `
        .dpd-pickup-button:disabled { background: #ccc !important; color: #666 !important; cursor: not-allowed !important; }
        .method-name.dpd-with-icon { position: relative; padding-left: 56px; }
        .method-name .dpd-method-icon-left {
          position: absolute; left: 12px; top: 50%; transform: translateY(-50%);
          width: 42px; height: 42px; object-fit: contain; pointer-events: none;
        }
        .shipping-method-next-step-button[disabled] { opacity: .6; cursor: not-allowed; pointer-events: none; }
      `;
      document.head.appendChild(style);
    }

    if (!document.getElementById('dpd-modal-overlay')) {
      const html = `
        <div id="dpd-modal-overlay" style="display:none;position:fixed;z-index:9999;top:0;left:0;width:100vw;height:100vh;background-color:rgba(0,0,0,0.6);justify-content:center;align-items:center;">
          <div style="position:relative;width:95%;height:95%;background:white;border-radius:6px;overflow:hidden;">
            <button id="dpd-modal-close" style="position:absolute;top:10px;right:10px;z-index:10;">Zavřít</button>
            <iframe id="dpd-widget-iframe" style="width:100%;height:100%;border:none;"></iframe>
          </div>
        </div>`;
      document.body.insertAdjacentHTML('beforeend', html);
      document.getElementById('dpd-modal-close').addEventListener('click', () => {
        document.getElementById('dpd-modal-overlay').style.display = 'none';
      });
    }

    const radios = (scope || document).querySelectorAll('input[type="radio"][name="shippingoption"]');
    radios.forEach(radio => {
      const li = radio.closest('li');
      if (!li || li.dataset.dpdEnhanced === '1') return;

      const { system } = parseShippingValue(radio.value);
      const isDpd = system.toLowerCase().includes('shipping.dpd');

      if (isDpd && WIDGET_MODE === 1 && ICON_URL) {
        const methodName = li.querySelector('.method-name');
        if (methodName) {
          methodName.classList.add('dpd-with-icon');
          methodName.querySelector('.dpd-method-icon-left')?.remove();
          const img = new Image();
          img.src = ICON_URL;
          img.alt = "DPD";
          img.className = "dpd-method-icon-left";
          methodName.insertAdjacentElement('afterbegin', img);
        }
      }

      if (isDpd && isDpdPickup(radio.value) && !li.querySelector('.dpd-pickup-button')) {
        const button = document.createElement('button');
        button.innerText = 'Vybrat výdejní místo';
        button.className = 'dpd-pickup-button';
        button.disabled = !radio.checked;
        button.style.cssText = `
          margin-top: 10px; padding: 6px 12px; background: #e00000; color: white;
          border: none; font-size: 0.9em; border-radius: 4px; cursor: pointer;
        `;
        button.addEventListener('click', e => {
          e.preventDefault();

          const shippingText = li.innerText.toLowerCase();
          let isoCountry = 'CZ';
          if (shippingText.includes('slovensko') || shippingText.includes('(sk)') || shippingText.includes('slovakia')) {
            isoCountry = 'SK';
          }

          const disableLockers = getCurrentDisableLockers(); // true = skrýt boxy

          const iframe = document.getElementById('dpd-widget-iframe');
          iframe.src = `https://api.dpd.cz/widget/latest/index.html?disableLockers=${disableLockers}&enabledCountries=${isoCountry}&language=cs`;
          document.getElementById('dpd-modal-overlay').style.display = 'flex';
        });
        li.querySelector('.method-name')?.appendChild(button);
      }

      if (radio.dataset.dpdWired !== '1') {
        radio.dataset.dpdWired = '1';
        radio.addEventListener('change', () => {
          document.querySelectorAll('.dpd-pickup-button').forEach(btn => {
            const rli = btn.closest('li');
            const rinput = rli?.querySelector('input[type="radio"][name="shippingoption"]');
            btn.disabled = !rinput?.checked;
          });

          handleDpdSelection(radio);
        });
      }

      li.dataset.dpdEnhanced = '1';
    });

    updateNextButtonState();
  }

  function getCurrentDisableLockers() {
    let result = false;

    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (!selected) return result;

    const { name, system } = parseShippingValue(selected.value);
    if (!system.toLowerCase().includes('shipping.dpd')) return result;

    const meta = DPD_META.get(name.toLowerCase());
    if (!meta) return result;

    result = !Boolean(meta.allowBoxDelivery);
    return result;
  }

  function whenShippingOptionsReadyOnce(timeoutMs = 8000) {
    if (document.querySelector('input[name="shippingoption"]')) return Promise.resolve();
    return new Promise(resolve => {
      const target = document.getElementById('checkout-shipping-method-load') || document.body;
      const to = setTimeout(() => { obs.disconnect(); resolve(); }, timeoutMs);
      const obs = new MutationObserver(() => {
        if (document.querySelector('input[name="shippingoption"]')) {
          clearTimeout(to);
          obs.disconnect();
          resolve();
        }
      });
      obs.observe(target, { childList: true, subtree: true });
    });
  }

  const opcContainer = document.getElementById('checkout-shipping-method-load');
  if (opcContainer && !opcContainer.dataset.dpdObserveAttached) {
    opcContainer.dataset.dpdObserveAttached = '1';
    const mo = new MutationObserver(() => { initDpdUI(opcContainer); });
    mo.observe(opcContainer, { childList: true, subtree: true });
  }

  (async () => {
    await loadDpdMetaOnce();
    await whenShippingOptionsReadyOnce();
    initDpdUI(document);

    const selected = document.querySelector('input[name="shippingoption"]:checked');
    if (selected) {
      const { system } = parseShippingValue(selected.value);
      const isDpd2 = system.toLowerCase().includes('shipping.dpd');
      const isPickup = isDpdPickup(selected.value);

      if (isDpd2 && !isPickup) {
        handleDpdSelection(selected);
      } else {
        updateNextButtonState();
      }
    } else {
      updateNextButtonState();
    }
  })();

  window.addEventListener("message", function (event) {
    if (!event.data?.dpdWidget || !event.data.dpdWidget.pickupPointResult) return;

    const dataString = event.data.dpdWidget.pickupPointResult;
    const parts = dataString.split(',').map(p => p.trim());
    const id = parts[0];
    const street = parts[1];
    const addressPart = parts[2];

    let zip = "", city = "", country = "";
    if (addressPart) {
      const addrTokens = addressPart.trim().split(" ");
      zip = addrTokens[0];
      country = addrTokens[addrTokens.length - 1];
      city = addrTokens.slice(1, -1).join(" ");
    }

    const formattedAddress = `${street}, ${zip} ${city} ${country}`;
    const name = `Výdejní místo, ${formattedAddress}`;

    document.getElementById('dpd-modal-overlay').style.display = 'none';

    document.querySelectorAll('.dpd-pickup-button').forEach(btn => {
      const li = btn.closest('li');
      const radio = li?.querySelector('input[type="radio"]');
      if (radio?.checked) {
        li.querySelector('.dpd-selected-address')?.remove();
        const div = document.createElement('div');
        div.className = 'dpd-selected-address';
        div.style.marginTop = '5px';
        div.style.fontSize = '0.9em';
        div.style.color = '#333';
        div.textContent = formattedAddress;
        btn.insertAdjacentElement('afterend', div);
      }
    });

    updateNextButtonState();

    const pickupPointModel = {
      id, name, street, city, zip, country,
      carrierId: 0,
      carrierPickupPointId: id,
      formattedAddress,
      isPickupPoint: true
    };

    const formData = new URLSearchParams();
    for (const key in pickupPointModel)
      formData.append(key.charAt(0).toUpperCase() + key.slice(1), pickupPointModel[key]);

    fetch('/DPDCheckout/SavePickupPoint', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: formData.toString()
    })
      .then(r => r.json())
      .then(res => {
        if (!res?.success) console.warn("Chyba při ukládání DPD výdejního místa:", res);
      })
      .catch(err => console.error("DPD widget fetch error:", err));
  }, false);

});
