﻿(function () {
  // ---- CONFIG ------------------------------------------------------------
  const ARES_ENDPOINT = '/AresForm/GetByIco';

  // ---- HELPERS -----------------------------------------------------------
  const q = (s, r) => (r || document).querySelector(s);
  const qa = (s, r) => Array.from((r || document).querySelectorAll(s));
  const deaccent = (s) => (s || '').toString().normalize('NFD').replace(/[\u0300-\u036f]/g, '').toLowerCase().trim();

  function findAddressBlocks() {
    return qa('.edit-address');
  }

  function rowBySuffix(container, suffix) {
    const el = q(`[name$='${suffix}'], #${suffix}`, container);
    return el ? el.closest('.inputs') : null;
  }

  function findCompanyRow(c) { return rowBySuffix(c, 'Company'); }
  function findCountryRow(c) { return rowBySuffix(c, 'CountryId'); }
  function findAddress1Row(c) { return rowBySuffix(c, 'Address1'); }

  function findIcoRow(c) {
    let rows = qa('.inputs.custom-attributes', c);
    for (const r of rows) {
      const lbl = r.querySelector('label');
      if (lbl && deaccent(lbl.textContent).includes('ico')) return r;
    }
    rows = qa('.inputs', c);
    for (const r of rows) {
      const lbl = r.querySelector('label');
      if (lbl && deaccent(lbl.textContent).includes('ico')) return r;
    }
    return null;
  }

  function addAresButton(icoRow) {
    if (!icoRow || icoRow.querySelector('.ares-inline .ares-lookup-btn')) return;

    const icoInput = icoRow.querySelector('input, textarea, select');
    if (!icoInput) return;

    const sib = icoInput.nextElementSibling;
    const isValidation = sib && (sib.matches('.field-validation-valid, .field-validation-error') || sib.hasAttribute('data-valmsg-for'));

    let inline = icoInput.parentElement;
    if (!inline || !inline.classList.contains('ares-inline')) {
      inline = document.createElement('span');
      inline.className = 'ares-inline';

      if (isValidation) icoRow.insertBefore(inline, sib);
      else icoInput.insertAdjacentElement('afterend', inline);

      inline.appendChild(icoInput);

      while (inline.nextSibling && inline.nextSibling !== sib &&
        inline.nextSibling.nodeType === 1 &&
        inline.nextSibling.classList.contains('required')) {
        inline.appendChild(inline.nextSibling);
      }
    }

    const wrap = document.createElement('span');
    wrap.className = 'ares-lookup-wrap';

    const btn = document.createElement('button');
    btn.type = 'button';
    btn.className = 'ares-lookup-btn';
    btn.textContent = 'Vyhledat v ARES';

    wrap.appendChild(btn);
    inline.appendChild(wrap);

    const run = typeof runLookupFor === 'function'
      ? () => runLookupFor(icoRow.closest('.edit-address') || document, icoInput, btn)
      : () => runLookup(icoInput, btn);

    icoInput.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') { e.preventDefault(); run(); }
    });
    btn.addEventListener('click', run);
  }

  function ensureIcoPlacement(container) {
    const icoRow = findIcoRow(container);
    if (!icoRow) return false;

    addAresButton(icoRow);

    const companyRow = findCompanyRow(container);
    const countryRow = findCountryRow(container);
    const address1Row = findAddress1Row(container);

    const parent = icoRow.parentElement;
    if (!parent) return false;

    let correct = false;
    if (companyRow && companyRow.nextElementSibling === icoRow) correct = true;
    else if (!companyRow && countryRow && icoRow === countryRow.previousElementSibling) correct = true;
    else if (!companyRow && !countryRow && address1Row && icoRow === address1Row.previousElementSibling) correct = true;

    if (correct) return true;

    if (companyRow) {
      parent.insertBefore(icoRow, companyRow.nextElementSibling);
    } else if (countryRow) {
      parent.insertBefore(icoRow, countryRow);
    } else if (address1Row) {
      parent.insertBefore(icoRow, address1Row);
    } else {
      parent.appendChild(icoRow);
    }
    return true;
  }

  function tryFindVatInContainer(container) {
    const rows = qa('.inputs.custom-attributes, .inputs', container);
    for (const r of rows) {
      const lbl = r.querySelector('label');
      if (!lbl) continue;
      const t = deaccent(lbl.textContent);
      if (t.includes('dic') || t.includes('vat')) {
        const input = r.querySelector('input, textarea, select');
        if (input && input.value) return input.value.trim();
      }
    }
    return '';
  }

  function setIfExistsIn(container, suffix, value) {
    const el = q(`[name$='${suffix}'], #${suffix}`, container);
    if (!el) return;

    if (el.tagName === 'SELECT') {
      const v = (value || '').toString().trim();
      const opt = Array.from(el.options).find(o =>
        o.value.toString().trim().toLowerCase() === v.toLowerCase() ||
        o.textContent.trim().toLowerCase() === v.toLowerCase());
      if (opt) el.value = opt.value;
      el.dispatchEvent(new Event('change', { bubbles: true }));
      return;
    }

    el.value = value || '';
    el.dispatchEvent(new Event('input', { bubbles: true }));
    el.dispatchEvent(new Event('change', { bubbles: true }));
  }

  function fillFromAresIn(container, data) {
    const streetCombined = [data.street || '', data.houseNumber || ''].filter(Boolean).join(' ').trim();

    setIfExistsIn(container, 'Company', data.companyName || '');
    setIfExistsIn(container, 'Address1', streetCombined);
    setIfExistsIn(container, 'City', data.city || '');
    setIfExistsIn(container, 'ZipPostalCode', data.zipPostalCode || '');

    if (data.countryId != null) {
      setIfExistsIn(container, 'CountryId', String(data.countryId));
    } else if (data.countryIso2) {
      trySetCountryByIso2In(container, data.countryIso2);
    }
  }


  function toBool(v) {
    if (typeof v === 'boolean') return v;
    if (typeof v === 'string') return v.toLowerCase() === 'true';
    return !!v;
  }

  async function runLookupFor(container, icoInput, btn) {
    const ico = (icoInput.value || '').replace(/\s/g, '');
    const vat = tryFindVatInContainer(container);

    const params = new URLSearchParams();
    if (ico) params.set('ico', ico);
    if (!ico && vat) params.set('vat', vat);

    const url = `${ARES_ENDPOINT}?${params.toString()}`;

    btn.disabled = true;
    btn.classList.add('is-loading');

    try {
      const res = await fetch(url, { method: 'GET', headers: { 'Accept': 'application/json' } });
      if (!res.ok) throw new Error('Chybná odpověď serveru');

      const json = await res.json();

      const success = toBool(json?.success ?? json?.Success);
      const data = json?.data ?? json?.Data;
      const message = json?.message ?? json?.Message ?? 'Subjekt nenalezen.';

      const looksLikeDto =
        data == null && (json?.companyName || json?.CompanyName || json?.ico || json?.ICO);

      if (!success && !looksLikeDto) {
        alert(message);
        return;
      }

      const payload = data ?? json;
      fillFromAresIn(container, payload || {});
    } catch (e) {
      console.error(e);
      alert('Nepodařilo se načíst data z ARES. Zkuste to prosím znovu.');
    } finally {
      btn.disabled = false;
      btn.classList.remove('is-loading');
    }
  }


  function processAll() {
    let foundAny = false;
    for (const container of findAddressBlocks()) {
      if (ensureIcoPlacement(container)) foundAny = true;
    }
    return foundAny;
  }

  function init() {
    processAll();

    let tries = 0;
    const poll = setInterval(() => {
      tries++;
      processAll();
      if (tries > 80) clearInterval(poll);
    }, 300);

    const mo = new MutationObserver(() => processAll());
    mo.observe(document.body, { childList: true, subtree: true });
  }

  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', init);
  else init();
})();
